<?php

namespace App\Filament\Pages;

use App\Models\Setting;
use Filament\Forms;
use Filament\Pages\Page;
use Filament\Notifications\Notification;

class ManageCountdown extends Page
{
    protected static ?string $navigationIcon = 'heroicon-o-clock';

    protected static ?string $navigationGroup = 'Content Management';

    protected static ?string $navigationLabel = 'Countdown Timer';

    protected static ?int $navigationSort = 2;

    protected static string $view = 'filament.pages.manage-countdown';

    public $data = [];

    public function mount()
    {
        $endTime = Setting::get('countdown_end_time', now()->addDays(7)->toDateTimeString());
        $dt = new \Carbon\Carbon($endTime);

        $this->form->fill([
            'countdown_title' => Setting::get('countdown_title', 'Season Ending Sale!'),
            'countdown_date' => $dt->format('Y-m-d'),
            'countdown_time' => $dt->format('H:i:s'),
            'countdown_is_visible' => (bool)Setting::get('countdown_is_visible', false),
            'countdown_bg_color' => Setting::get('countdown_bg_color', '#dc2626'),
            'countdown_text_color' => Setting::get('countdown_text_color', '#ffffff'),
        ]);
    }

    protected function getFormSchema(): array
    {
        return [
            Forms\Components\Card::make()
            ->schema([
                Forms\Components\Toggle::make('countdown_is_visible')
                ->label('Show Countdown Timer'),
                Forms\Components\TextInput::make('countdown_title')
                ->label('Timer Title')
                ->required()
                ->placeholder('e.g. Flash Sale Ends In:'),
                Forms\Components\Grid::make(2)
                ->schema([
                    Forms\Components\DatePicker::make('countdown_date')
                    ->label('End Date')
                    ->required(),
                    Forms\Components\TimePicker::make('countdown_time')
                    ->label('End Time')
                    ->required()
                    ->withoutSeconds(),
                ]),
                Forms\Components\Grid::make(2)
                ->schema([
                    Forms\Components\ColorPicker::make('countdown_bg_color')
                    ->label('Background Color')
                    ->default('#dc2626'),
                    Forms\Components\ColorPicker::make('countdown_text_color')
                    ->label('Text Color')
                    ->default('#ffffff'),
                ]),
            ])
        ];
    }

    protected function getFormStatePath(): string
    {
        return 'data';
    }

    public function submit()
    {
        $data = $this->form->getState();

        // Combine date and time
        $endTime = $data['countdown_date'] . ' ' . $data['countdown_time'];

        // Remove individual components from data before loop
        unset($data['countdown_date'], $data['countdown_time']);

        // Save combined value
        Setting::set('countdown_end_time', $endTime, 'countdown');

        foreach ($data as $key => $value) {
            Setting::set($key, $value, 'countdown');
        }

        Notification::make()
            ->title('Countdown timer settings saved!')
            ->success()
            ->send();
    }
}
