<?php

namespace App\Filament\Resources;

use App\Filament\Resources\BlogPostResource\Pages;
use App\Models\BlogPost;
use Filament\Forms;
use Filament\Resources\Form;
use Filament\Resources\Resource;
use Filament\Resources\Table;
use Filament\Tables;
use Illuminate\Support\Str;

class BlogPostResource extends Resource
{
    protected static ?string $model = BlogPost::class;
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?string $navigationGroup = 'Content Management';
    protected static ?string $navigationLabel = 'Blog Posts';
    protected static ?int $navigationSort = 1;
    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Group::make()->schema([
                Forms\Components\Card::make()->schema([
                    Forms\Components\TextInput::make('title')->required()->maxLength(255)
                        ->reactive()->afterStateUpdated(fn ($state, callable $set) => $set('slug', Str::slug($state))),
                    Forms\Components\TextInput::make('slug')->required()->disabled()->dehydrated()
                        ->unique(BlogPost::class, 'slug', ignoreRecord: true),
                    Forms\Components\RichEditor::make('content')->required()->columnSpanFull(),
                ])->columns(2),
            ])->columnSpan(['lg' => 2]),
            Forms\Components\Group::make()->schema([
                Forms\Components\Card::make()->schema([
                    Forms\Components\FileUpload::make('featured_image')->image()->directory('blog'),
                    Forms\Components\Select::make('user_id')->relationship('author','name')->required()->searchable()->preload()->label('Author'),
                    Forms\Components\DateTimePicker::make('published_at')->label('Publish Date'),
                ]),
            ])->columnSpan(['lg' => 1]),
        ])->columns(3);
    }

    public static function table(Table $table): Table
    {
        return $table->columns([
            Tables\Columns\ImageColumn::make('featured_image')->label('Image'),
            Tables\Columns\TextColumn::make('title')->searchable()->sortable()->limit(40),
            Tables\Columns\TextColumn::make('author.name')->label('Author')->searchable(),
            Tables\Columns\TextColumn::make('published_at')->dateTime('M d, Y')->sortable(),
            Tables\Columns\TextColumn::make('created_at')->dateTime('M d, Y')->sortable(),
        ])->defaultSort('created_at','desc')
        ->actions([Tables\Actions\EditAction::make(),Tables\Actions\DeleteAction::make()])->bulkActions([Tables\Actions\DeleteBulkAction::make()]);
    }

    public static function getPages(): array
    {
        return ['index'=>Pages\ListBlogPosts::route('/'),'create'=>Pages\CreateBlogPost::route('/create'),'edit'=>Pages\EditBlogPost::route('/{record}/edit')];
    }
}
