<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CouponResource\Pages;
use App\Models\Coupon;
use Filament\Forms;
use Filament\Resources\Form;
use Filament\Resources\Resource;
use Filament\Resources\Table;
use Filament\Tables;

class CouponResource extends Resource
{
    protected static ?string $model = Coupon::class;

    protected static ?string $navigationGroup = 'Marketing';

    protected static ?string $navigationLabel = 'Coupons / Discounts';

    protected static ?int $navigationSort = 1;

    protected static ?string $navigationIcon = 'heroicon-o-ticket';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Card::make()
                    ->schema([
                        Forms\Components\TextInput::make('code')
                            ->required()
                            ->unique(ignoreRecord: true)
                            ->maxLength(255)
                            ->helperText('Use uppercase letters for better readability (e.g., SUMMER50).'),
                        Forms\Components\Select::make('type')
                            ->options([
                                'percentage' => 'Percentage (%)',
                                'fixed' => 'Fixed Amount ($)',
                            ])
                            ->required()
                            ->default('percentage'),
                        Forms\Components\TextInput::make('value')
                            ->numeric()
                            ->required()
                            ->label('Discount Value')
                            ->helperText('For percentage, enter 50 for 50%. For fixed, enter 10 for $10.'),
                        Forms\Components\TextInput::make('min_order_amount')
                            ->numeric()
                            ->label('Min. Order Amount')
                            ->helperText('Minimum cart total to apply this coupon.'),
                        Forms\Components\TextInput::make('usage_limit')
                            ->numeric()
                            ->label('Usage Limit')
                            ->helperText('Total times this coupon can be used.'),
                        Forms\Components\DateTimePicker::make('expires_at')
                            ->label('Expiry Date'),
                        Forms\Components\Toggle::make('is_active')
                            ->label('Active')
                            ->default(true),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('code')->searchable()->sortable()->weight('bold'),
                Tables\Columns\BadgeColumn::make('type')
                    ->colors([
                        'primary' => 'percentage',
                        'success' => 'fixed',
                    ]),
                Tables\Columns\TextColumn::make('value')
                    ->formatStateUsing(fn ($record) => $record->type === 'percentage' ? $record->value . '%' : '$' . number_format($record->value, 2)),
                Tables\Columns\TextColumn::make('usage_limit')
                    ->label('Limit'),
                Tables\Columns\TextColumn::make('used_count')
                    ->label('Used'),
                Tables\Columns\TextColumn::make('expires_at')
                    ->dateTime('M d, Y')
                    ->sortable(),
                Tables\Columns\BooleanColumn::make('is_active')->label('Active'),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCoupons::route('/'),
            'create' => Pages\CreateCoupon::route('/create'),
            'edit' => Pages\EditCoupon::route('/{record}/edit'),
        ];
    }
}
