<?php

namespace App\Filament\Resources;

use App\Filament\Resources\MediaFileResource\Pages;
use App\Models\MediaFile;
use Filament\Forms;
use Filament\Resources\Form;
use Filament\Resources\Resource;
use Filament\Resources\Table;
use Filament\Tables;
use Livewire\TemporaryUploadedFile;

class MediaFileResource extends Resource
{
    protected static ?string $model = MediaFile::class;
    protected static ?string $navigationGroup = 'Content Management';
    protected static ?string $navigationLabel = 'Media Library';
    protected static ?int $navigationSort = 2;
    protected static ?string $navigationIcon = 'heroicon-o-collection';

    // protected static bool $shouldRegisterNavigation = false;

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Card::make()->schema([
                Forms\Components\TextInput::make('name')
                ->required()
                ->maxLength(255)
                ->placeholder('Enter a name for this file'),

                Forms\Components\Select::make('file_type')
                ->label('Type')
                ->options([
                    'image' => 'Image',
                    'video' => 'Video',
                    'document' => 'Document',
                    'audio' => 'Audio',
                ])
                ->default('image')
                ->disabled() // Auto-detected
                ->dehydrated() // Send value even if disabled
                ->required(),

                Forms\Components\FileUpload::make('file_path')
                ->label('File')
                ->required()
                ->disk('public') // Force public disk
                ->directory('media')
                ->visibility('public')
                ->maxSize(51200) // 50MB
                ->columnSpanFull()
                ->preserveFilenames(),

                Forms\Components\Select::make('folder')
                ->label('Folder')
                ->options([
                    'general' => 'General',
                    'products' => 'Products',
                    'banners' => 'Banners',
                    'blog' => 'Blog',
                    'pages' => 'Pages',
                ])
                ->default('general'),

                Forms\Components\Placeholder::make('file_size_display')
                ->label('File Size')
                ->content(fn(?MediaFile $record): string => $record ?self::formatFileSize($record->file_size) : 'Will be calculated on upload')
                ->visibleOn('edit'),
            ])->columns(2),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table->columns([
            Tables\Columns\ImageColumn::make('file_path')
            ->label('Preview')
            ->visibility('public')
            ->circular(),
            Tables\Columns\TextColumn::make('name')
            ->searchable()
            ->sortable()
            ->weight('bold'),
            Tables\Columns\BadgeColumn::make('file_type')
            ->label('Type')
            ->colors([
                'primary' => 'image',
                'warning' => 'video',
                'secondary' => 'document',
                'success' => 'audio',
            ]),
            Tables\Columns\BadgeColumn::make('folder')
            ->colors([
                'primary' => 'general',
                'success' => 'products',
                'warning' => 'banners',
                'info' => 'blog',
            ]),
            Tables\Columns\TextColumn::make('file_size')
            ->label('Size')
            ->formatStateUsing(fn($state) => self::formatFileSize($state)),
            Tables\Columns\TextColumn::make('created_at')
            ->dateTime('M d, Y')
            ->sortable(),
        ])->defaultSort('created_at', 'desc')
            ->filters([
            Tables\Filters\SelectFilter::make('file_type')
            ->label('Type')
            ->options([
                'image' => 'Image',
                'video' => 'Video',
                'document' => 'Document',
            ]),
            Tables\Filters\SelectFilter::make('folder')
            ->options([
                'general' => 'General',
                'products' => 'Products',
                'banners' => 'Banners',
                'blog' => 'Blog',
            ]),
        ])
            ->actions([
            Tables\Actions\EditAction::make(),
            Tables\Actions\DeleteAction::make(),
        ])
            ->bulkActions([Tables\Actions\DeleteBulkAction::make()]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListMediaFiles::route('/'),
            'create' => Pages\CreateMediaFile::route('/create'),
            'edit' => Pages\EditMediaFile::route('/{record}/edit'),
        ];
    }

    protected static function formatFileSize(?int $bytes): string
    {
        if (!$bytes)
            return '0 B';
        $units = ['B', 'KB', 'MB', 'GB'];
        $i = floor(log($bytes, 1024));
        return round($bytes / pow(1024, $i), 2) . ' ' . $units[$i];
    }
}
