<?php

namespace App\Filament\Resources;

use App\Filament\Resources\OrderResource\Pages;
use App\Models\Order;
use Filament\Forms;
use Filament\Resources\Form;
use Filament\Resources\Resource;
use Filament\Resources\Table;
use Filament\Tables;

class OrderResource extends Resource
{
    protected static ?string $model = Order::class;
    protected static ?string $recordTitleAttribute = 'number';
    protected static ?string $navigationGroup = 'Sales & CRM';
    protected static ?string $navigationLabel = 'Orders';
    protected static ?int $navigationSort = 1;
    protected static ?string $navigationIcon = 'heroicon-o-shopping-bag';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Group::make()->schema([
                Forms\Components\Card::make()->schema([
                    Forms\Components\TextInput::make('number')->label('Order Number')->disabled()->dehydrated(),
                    Forms\Components\Select::make('user_id')->relationship('user', 'name')->searchable()->preload()->label('Customer'),
                    Forms\Components\Textarea::make('shipping_address')->rows(3)->columnSpanFull(),
                    Forms\Components\Textarea::make('notes')->rows(2)->columnSpanFull(),
                ])->columns(2),
            ])->columnSpan(['lg' => 2]),
            Forms\Components\Group::make()->schema([
                Forms\Components\Card::make()->schema([
                    Forms\Components\Select::make('status')->options([
                        'pending' => 'Pending', 'confirmed' => 'Confirmed', 'processing' => 'Processing', 'shipped' => 'Shipped', 'delivered' => 'Delivered', 'cancelled' => 'Cancelled', 'returned' => 'Returned', 'refunded' => 'Refunded',
                    ])->required()->default('pending'),
                    Forms\Components\Select::make('payment_status')->options([
                        'pending' => 'Pending', 'paid' => 'Paid', 'failed' => 'Failed', 'refunded' => 'Refunded',
                    ])->required()->default('pending'),
                    Forms\Components\TextInput::make('total_price')->numeric()->prefix('$')->disabled()->dehydrated(),
                ]),
            ])->columnSpan(['lg' => 1]),
        ])->columns(3);
    }

    public static function table(Table $table): Table
    {
        return $table->columns([
            Tables\Columns\TextColumn::make('number')->label('Order #')->searchable()->sortable(),
            Tables\Columns\TextColumn::make('user.name')->label('Customer')->searchable()->sortable(),
            Tables\Columns\BadgeColumn::make('status')->colors([
                'warning' => 'pending', 'primary' => fn($state): bool => in_array($state, ['confirmed', 'processing']),
                'success' => 'delivered', 'danger' => fn($state): bool => in_array($state, ['cancelled', 'returned', 'refunded']),
            ]),
            Tables\Columns\BadgeColumn::make('payment_status')->colors([
                'warning' => 'pending', 'success' => 'paid', 'danger' => fn($state): bool => in_array($state, ['failed', 'refunded']),
            ]),
            Tables\Columns\TextColumn::make('total_price')->money('usd')->sortable(),
            Tables\Columns\TextColumn::make('items_count')->counts('items')->label('Items'),
            Tables\Columns\TextColumn::make('created_at')->dateTime('M d, Y H:i')->sortable()->label('Date'),
        ])->defaultSort('created_at', 'desc')
            ->filters([
            Tables\Filters\SelectFilter::make('status')->options(['pending' => 'Pending', 'confirmed' => 'Confirmed', 'shipped' => 'Shipped', 'delivered' => 'Delivered', 'cancelled' => 'Cancelled']),
            Tables\Filters\SelectFilter::make('payment_status')->options(['pending' => 'Pending', 'paid' => 'Paid', 'failed' => 'Failed']),
        ])->actions([
            Tables\Actions\EditAction::make(),
            Tables\Actions\Action::make('print_invoice')
            ->label('Invoice')
            ->icon('heroicon-o-printer')
            ->url(fn(Order $record) => route('admin.invoices.print', $record))
            ->openUrlInNewTab(),
        ])->bulkActions([Tables\Actions\DeleteBulkAction::make()]);
    }

    public static function getRelations(): array
    {
        return [];
    }
    public static function getPages(): array
    {
        return ['index' => Pages\ListOrders::route('/'), 'create' => Pages\CreateOrder::route('/create'), 'edit' => Pages\EditOrder::route('/{record}/edit')];
    }
}
