<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ProductResource\Pages;
use App\Models\Product;
use App\Models\Category;
use Filament\Forms;
use Filament\Resources\Form;
use Filament\Resources\Resource;
use Filament\Resources\Table;
use Filament\Tables;
use Illuminate\Support\Str;

class ProductResource extends Resource
{
    protected static ?string $model = Product::class;
    protected static ?string $recordTitleAttribute = 'name';
    protected static ?string $navigationGroup = 'Shop Management';
    protected static ?string $navigationLabel = 'Products';
    protected static ?int $navigationSort = 1;
    protected static ?string $navigationIcon = 'heroicon-o-cube';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Group::make()
                    ->schema([
                        Forms\Components\Card::make()
                            ->schema([
                                Forms\Components\TextInput::make('name')
                                    ->required()
                                    ->maxLength(255)
                                    ->reactive()
                                    ->afterStateUpdated(fn ($state, callable $set) => $set('slug', Str::slug($state))),
                                Forms\Components\TextInput::make('slug')
                                    ->required()
                                    ->maxLength(255)
                                    ->disabled()
                                    ->dehydrated()
                                    ->unique(Product::class, 'slug', ignoreRecord: true),
                                Forms\Components\RichEditor::make('description')
                                    ->columnSpanFull(),
                                Forms\Components\TextInput::make('short_description')
                                    ->maxLength(500),
                            ])->columns(2),
                        Forms\Components\Card::make()
                            ->schema([
                                Forms\Components\FileUpload::make('image')
                                    ->image()
                                    ->directory('products')
                                    ->columnSpanFull()
                                    ->label('Product Image'),
                            ]),
                        Forms\Components\Card::make()
                            ->schema([
                                Forms\Components\TextInput::make('price')
                                    ->required()
                                    ->numeric()
                                    ->prefix('$')
                                    ->maxValue(999999),
                                Forms\Components\TextInput::make('sale_price')
                                    ->numeric()
                                    ->prefix('$')
                                    ->maxValue(999999),
                                Forms\Components\DateTimePicker::make('sale_price_start')
                                    ->label('Sale Start'),
                                Forms\Components\DateTimePicker::make('sale_price_end')
                                    ->label('Sale End'),
                            ])->columns(2),
                    ])->columnSpan(['lg' => 2]),
                Forms\Components\Group::make()
                    ->schema([
                        Forms\Components\Card::make()
                            ->schema([
                                Forms\Components\Select::make('parent_category')
                                    ->label('Category')
                                    ->options(function () {
                                        return Category::whereNull('parent_id')
                                            ->where('is_active', true)
                                            ->orderBy('name')
                                            ->pluck('name', 'id');
                                    })
                                    ->reactive()
                                    ->afterStateHydrated(function ($state, callable $set, $record) {
                                        if ($record && $record->category_id) {
                                            $cat = Category::find($record->category_id);
                                            if ($cat) {
                                                // Trace up to root
                                                $current = $cat;
                                                while ($current->parent_id !== null) {
                                                    $current = Category::find($current->parent_id);
                                                    if (!$current) break;
                                                }
                                                $set('parent_category', $current ? $current->id : null);
                                            }
                                        }
                                    })
                                    ->afterStateUpdated(function (callable $set) {
                                        $set('category_id', null);
                                    })
                                    ->required()
                                    ->searchable()
                                    ->dehydrated(false)
                                    ->placeholder('Select parent category...'),
                                Forms\Components\Select::make('category_id')
                                    ->label('Sub Category')
                                    ->options(function (callable $get) {
                                        $parentId = $get('parent_category');
                                        if (!$parentId) {
                                            return [];
                                        }
                                        $options = [];
                                        $children = Category::where('parent_id', $parentId)
                                            ->where('is_active', true)
                                            ->orderBy('name')
                                            ->get();
                                        if ($children->isEmpty()) {
                                            // No subcategories, allow selecting parent itself
                                            $parent = Category::find($parentId);
                                            return $parent ? [$parent->id => $parent->name] : [];
                                        }
                                        foreach ($children as $child) {
                                            $grandchildren = Category::where('parent_id', $child->id)
                                                ->where('is_active', true)
                                                ->orderBy('name')
                                                ->get();
                                            if ($grandchildren->isNotEmpty()) {
                                                $options[$child->id] = $child->name;
                                                foreach ($grandchildren as $gc) {
                                                    $options[$gc->id] = '-- ' . $gc->name;
                                                }
                                            } else {
                                                $options[$child->id] = $child->name;
                                            }
                                        }
                                        return $options;
                                    })
                                    ->required()
                                    ->searchable()
                                    ->placeholder('Select sub category...'),
                            ]),
                        Forms\Components\Card::make()
                            ->schema([
                                Forms\Components\Select::make('brand_id')
                                    ->relationship('brand', 'name')
                                    ->searchable()
                                    ->preload(),
                            ]),
                        Forms\Components\Card::make()
                            ->schema([
                                Forms\Components\TextInput::make('sku')
                                    ->label('SKU')
                                    ->maxLength(50)
                                    ->unique(Product::class, 'sku', ignoreRecord: true),
                                Forms\Components\TextInput::make('stock')
                                    ->numeric()
                                    ->default(0)
                                    ->minValue(0),
                                Forms\Components\TextInput::make('low_stock_warning')
                                    ->numeric()
                                    ->default(10)
                                    ->helperText('Alert when stock falls below'),
                            ]),
                        Forms\Components\Card::make()
                            ->schema([
                                Forms\Components\Toggle::make('is_active')
                                    ->label('Published')
                                    ->default(true),
                                Forms\Components\Toggle::make('is_featured')
                                    ->label('Featured'),
                                Forms\Components\Toggle::make('on_sale')
                                    ->label('On Sale'),
                            ]),
                    ])->columnSpan(['lg' => 1]),
            ])->columns(3);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ImageColumn::make('image')->label('Image'),
                Tables\Columns\TextColumn::make('name')->searchable()->sortable()->limit(30),
                Tables\Columns\TextColumn::make('category.name')->label('Category')->searchable(),
                Tables\Columns\TextColumn::make('price')->money('usd')->sortable(),
                Tables\Columns\TextColumn::make('sale_price')->money('usd'),
                Tables\Columns\TextColumn::make('stock')->sortable(),
                Tables\Columns\BooleanColumn::make('is_active')->label('Active'),
                Tables\Columns\BooleanColumn::make('is_featured')->label('Featured'),
                Tables\Columns\TextColumn::make('created_at')->dateTime('M d, Y')->sortable(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                Tables\Filters\SelectFilter::make('category')->relationship('category', 'name'),
                Tables\Filters\TernaryFilter::make('is_active')->label('Published'),
                Tables\Filters\TernaryFilter::make('is_featured')->label('Featured'),
                Tables\Filters\TernaryFilter::make('on_sale')->label('On Sale'),
            ])
            ->actions([Tables\Actions\EditAction::make(), Tables\Actions\DeleteAction::make()])
            ->bulkActions([Tables\Actions\DeleteBulkAction::make()]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListProducts::route('/'),
            'create' => Pages\CreateProduct::route('/create'),
            'edit' => Pages\EditProduct::route('/{record}/edit'),
        ];
    }
}