<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ReviewResource\Pages;
use App\Models\Review;
use Filament\Forms;
use Filament\Resources\Form;
use Filament\Resources\Resource;
use Filament\Resources\Table;
use Filament\Tables;

class ReviewResource extends Resource
{
    protected static ?string $model = Review::class;
    protected static ?string $navigationGroup = 'Shop Management';
    protected static ?string $navigationLabel = 'Review Management';
    protected static ?int $navigationSort = 5;
    protected static ?string $navigationIcon = 'heroicon-o-star';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Card::make()->schema([
                Forms\Components\Select::make('product_id')
                ->relationship('product', 'name')
                ->required()
                ->searchable()
                ->preload()
                ->label('Product'),
                Forms\Components\Select::make('user_id')
                ->relationship('user', 'name')
                ->required()
                ->searchable()
                ->preload()
                ->label('Customer'),
                Forms\Components\Select::make('rating')
                ->options([
                    1 => '⭐ 1 Star',
                    2 => '⭐⭐ 2 Stars',
                    3 => '⭐⭐⭐ 3 Stars',
                    4 => '⭐⭐⭐⭐ 4 Stars',
                    5 => '⭐⭐⭐⭐⭐ 5 Stars',
                ])
                ->required(),
                Forms\Components\Toggle::make('is_approved')
                ->label('Approved')
                ->default(false),
            ])->columns(2),
            Forms\Components\Card::make()->schema([
                Forms\Components\Textarea::make('comment')
                ->rows(3)
                ->label('Review Comment')
                ->columnSpanFull(),
            ]),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table->columns([
            Tables\Columns\TextColumn::make('product.name')
            ->label('Product')
            ->searchable()
            ->sortable()
            ->limit(25),
            Tables\Columns\TextColumn::make('user.name')
            ->label('Customer')
            ->searchable(),
            Tables\Columns\TextColumn::make('rating')
            ->sortable()
            ->formatStateUsing(fn($state) => str_repeat('⭐', $state)),
            Tables\Columns\TextColumn::make('comment')
            ->limit(40),
            Tables\Columns\BooleanColumn::make('is_approved')
            ->label('Approved'),
            Tables\Columns\TextColumn::make('created_at')
            ->dateTime('M d, Y')
            ->sortable(),
        ])->defaultSort('created_at', 'desc')
            ->filters([
            Tables\Filters\TernaryFilter::make('is_approved')
            ->label('Approved'),
            Tables\Filters\SelectFilter::make('rating')
            ->options([
                1 => '1 Star',
                2 => '2 Stars',
                3 => '3 Stars',
                4 => '4 Stars',
                5 => '5 Stars',
            ]),
        ])
            ->actions([
            Tables\Actions\EditAction::make(),
            Tables\Actions\DeleteAction::make(),
        ])
            ->bulkActions([Tables\Actions\DeleteBulkAction::make()]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListReviews::route('/'),
            'create' => Pages\CreateReview::route('/create'),
            'edit' => Pages\EditReview::route('/{record}/edit'),
        ];
    }
}
