<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TransactionResource\Pages;
use App\Models\Transaction;
use Filament\Forms;
use Filament\Resources\Form;
use Filament\Resources\Resource;
use Filament\Resources\Table;
use Filament\Tables;

class TransactionResource extends Resource
{
    protected static ?string $model = Transaction::class;
    protected static ?string $navigationGroup = 'Finance';
    protected static ?string $navigationLabel = 'Transactions';
    protected static ?int $navigationSort = 1;
    protected static ?string $navigationIcon = 'heroicon-o-currency-dollar';

    protected static bool $shouldRegisterNavigation = false;

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Card::make()->schema([
                Forms\Components\Select::make('account_id')
                ->relationship('account', 'name')
                ->required()
                ->searchable()
                ->preload()
                ->label('Account'),
                Forms\Components\Select::make('type')
                ->options([
                    'debit' => 'Debit (Money Out)',
                    'credit' => 'Credit (Money In)',
                ])
                ->required()
                ->default('credit'),
                Forms\Components\TextInput::make('amount')
                ->numeric()
                ->required()
                ->prefix('$')
                ->minValue(0.01),
                Forms\Components\DatePicker::make('transaction_date')
                ->required()
                ->default(now())
                ->label('Transaction Date'),
                Forms\Components\Textarea::make('description')
                ->rows(2)
                ->columnSpanFull()
                ->placeholder('Transaction description or notes'),
            ])->columns(2),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table->columns([
            Tables\Columns\TextColumn::make('id')
            ->label('#')
            ->sortable(),
            Tables\Columns\TextColumn::make('account.name')
            ->label('Account')
            ->searchable()
            ->sortable(),
            Tables\Columns\BadgeColumn::make('type')
            ->colors([
                'success' => 'credit',
                'danger' => 'debit',
            ])
            ->formatStateUsing(fn($state) => $state === 'credit' ? 'Credit' : 'Debit'),
            Tables\Columns\TextColumn::make('amount')
            ->money('usd')
            ->sortable()
            ->color(fn($record) => $record->type === 'credit' ? 'success' : 'danger'),
            Tables\Columns\TextColumn::make('description')
            ->limit(40)
            ->searchable(),
            Tables\Columns\TextColumn::make('transaction_date')
            ->date('M d, Y')
            ->sortable()
            ->label('Date'),
        ])->defaultSort('transaction_date', 'desc')
            ->filters([
            Tables\Filters\SelectFilter::make('type')
            ->options([
                'debit' => 'Debit',
                'credit' => 'Credit',
            ]),
            Tables\Filters\SelectFilter::make('account')
            ->relationship('account', 'name'),
        ])
            ->actions([
            Tables\Actions\EditAction::make(),
            Tables\Actions\DeleteAction::make(),
        ])
            ->bulkActions([Tables\Actions\DeleteBulkAction::make()]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTransactions::route('/'),
            'create' => Pages\CreateTransaction::route('/create'),
            'edit' => Pages\EditTransaction::route('/{record}/edit'),
        ];
    }
}
