<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CheckoutController extends Controller
{
    public function store(Request $request)
    {
        $validated = $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'email' => 'required|email',
            'phone' => 'required|string',
            'address' => 'required|string',
            'city' => 'required|string',
            'postal_code' => 'nullable|string',
            'payment_method' => 'required|string',
        ]);

        try {
            DB::beginTransaction();

            $total = 0;
            $orderItemsData = [];

            foreach ($validated['items'] as $item) {
                $product = \App\Models\Product::findOrFail($item['id']);
                $price = $product->price;
                $variationId = $item['variation_id'] ?? null;

                if ($variationId) {
                    $variation = \App\Models\ProductVariation::where('product_id', $product->id)
                        ->where('id', $variationId)
                        ->firstOrFail();

                    if ($variation->stock < $item['quantity']) {
                        throw new \Exception("Insufficient stock for {$product->name} ({$variation->value})");
                    }

                    $price += $variation->price_adjustment;
                    $variation->decrement('stock', $item['quantity']);
                }
                else {
                    if ($product->stock < $item['quantity']) {
                        throw new \Exception("Insufficient stock for {$product->name}");
                    }
                    $product->decrement('stock', $item['quantity']);
                }

                $total += $price * $item['quantity'];

                $orderItemsData[] = [
                    'product_id' => $product->id,
                    'variation_id' => $variationId,
                    'quantity' => $item['quantity'],
                    'unit_price' => $price,
                ];
            }

            // Create Order
            $order = \App\Models\Order::create([
                'user_id' => auth('sanctum')->id(), // Reliable ID capture
                'total_price' => $total,
                'status' => 'pending',
                'number' => 'ORD-' . strtoupper(uniqid()),
                'customer_first_name' => $validated['first_name'],
                'customer_last_name' => $validated['last_name'],
                'customer_email' => $validated['email'],
                'customer_phone' => $validated['phone'],
                'address' => $validated['address'],
                'city' => $validated['city'],
                'postal_code' => $validated['postal_code'] ?? null,
                'payment_method' => $validated['payment_method'],
                'payment_status' => 'pending',
                'shipping_address' => $validated['address'] . ', ' . $validated['city'] . ($validated['postal_code'] ? ', ' . $validated['postal_code'] : ''),
            ]);

            // Create Order Items
            foreach ($orderItemsData as $data) {
                \App\Models\OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $data['product_id'],
                    'variation_id' => $data['variation_id'],
                    'quantity' => $data['quantity'],
                    'unit_price' => $data['unit_price'],
                ]);
            }

            DB::commit();

            return response()->json([
                'message' => 'Order placed successfully!',
                'order_id' => $order->id,
            ], 201);

        }
        catch (\Exception $e) {
            DB::rollBack();
            \Illuminate\Support\Facades\Log::error('Checkout Error: ' . $e->getMessage());
            \Illuminate\Support\Facades\Log::error('Request Data: ' . json_encode($request->all()));
            \Illuminate\Support\Facades\Log::error($e->getTraceAsString());
            return response()->json(['message' => 'Order failed', 'error' => $e->getMessage()], 500);
        }
    }
}
