<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::where('is_active', true)->with('category');

        // Text Search
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Category Filter (supports hierarchical categories)
        if ($request->filled('category')) {
            $slug = $request->input('category');
            $cat = Category::where('slug', $slug)->first();

            if ($cat) {
                // Get all children recursively
                $categoryIds = collect([$cat->id]);

                // 1st level children
                $childIds = Category::where('parent_id', $cat->id)->pluck('id');
                $categoryIds = $categoryIds->merge($childIds);

                // 2nd level children
                if ($childIds->isNotEmpty()) {
                    $grandchildIds = Category::whereIn('parent_id', $childIds)->pluck('id');
                    $categoryIds = $categoryIds->merge($grandchildIds);
                }

                $query->whereIn('category_id', $categoryIds);
            }
            else {
                // Category not found by slug - return empty or handle error?
                // For now, let's force empty result if category param exists but is invalid
                $query->whereRaw('1 = 0');
            }
        }

        // Standard Filters
        if ($request->boolean('on_sale')) {
            $query->whereNotNull('sale_price');
        }
        if ($request->boolean('is_featured') || $request->has('featured')) {
            $query->where('is_featured', true);
        }
        if ($request->input('collection') === 'new') {
            $query->latest();
        }

        // Return flat array (not paginated) so frontend can handle pagination
        $products = $query->get();

        // Add ancestor category slugs for each product for client-side filtering
        // only needed if we want client-side nav, but good to keep
        return response()->json($products);
    }

    public function suggestions(Request $request)
    {
        $search = $request->input('query');
        if (!$search || strlen($search) < 2) {
            return response()->json([]);
        }

        // Search Products
        $products = Product::where('is_active', true)
            ->where('name', 'like', "%{$search}%")
            ->select('id', 'name', 'slug', 'image', 'price', 'sale_price')
            ->limit(5)
            ->get()
            ->map(function ($product) {
            return [
            'id' => $product->id,
            'name' => $product->name,
            'slug' => $product->slug,
            'image' => $product->image,
            'price' => $product->sale_price ?? $product->price,
            'type' => 'product'
            ];
        });

        // Search Categories
        $categories = Category::where('is_active', true)
            ->where('name', 'like', "%{$search}%")
            ->select('id', 'name', 'slug', 'image')
            ->limit(3)
            ->get()
            ->map(function ($category) {
            return [
            'id' => $category->id,
            'name' => $category->name,
            'slug' => $category->slug,
            'image' => $category->image,
            'type' => 'category'
            ];
        });

        return response()->json([
            'products' => $products,
            'categories' => $categories
        ]);
    }

    public function show($slug)
    {
        return response()->json(Product::where('slug', $slug)
            ->where('is_active', true)
            ->with(['category', 'variations'])
            ->firstOrFail());
    }
}