<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Product; // Assuming Product model exists
use Illuminate\Support\Facades\DB;

class WishlistController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();
        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        // Fetch products in the user's wishlist
        // distinct() is used in case of any duplicate data issues, though the unique constraint prevents it
        $products = DB::table('wishlists')
            ->join('products', 'wishlists.product_id', '=', 'products.id')
            ->where('wishlists.user_id', $user->id)
            ->select('products.*', 'wishlists.id as wishlist_id', 'wishlists.created_at as added_at')
            ->get();

        return response()->json($products);
    }

    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
        ]);

        $user = $request->user();
        $productId = $request->input('product_id');

        // Toggle logic: if exists, remove it; if not, add it. 
        // Or strictly add. The UI usually expects "toggle" or distinct "add"/"remove".
        // Let's implement strict "add" for this endpoint to be RESTful, 
        // and handle "remove" in destroy.
        // However, standard "heart" button behavior is often a toggle.
        // Let's stick to "Add" here. Frontend checks state.

        $exists = DB::table('wishlists')
            ->where('user_id', $user->id)
            ->where('product_id', $productId)
            ->exists();

        if ($exists) {
            return response()->json(['message' => 'Product already in wishlist'], 200);
        }

        DB::table('wishlists')->insert([
            'user_id' => $user->id,
            'product_id' => $productId,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return response()->json(['message' => 'Product added to wishlist'], 201);
    }

    public function destroy($productId, Request $request)
    {
        $user = $request->user();

        $deleted = DB::table('wishlists')
            ->where('user_id', $user->id)
            ->where('product_id', $productId)
            ->delete();

        if ($deleted) {
            return response()->json(['message' => 'Product removed from wishlist']);
        }

        return response()->json(['message' => 'Product not found in wishlist'], 404);
    }

    // Check if a specific product is in wishlist (helper for product page)
    public function check($productId, Request $request)
    {
        $user = $request->user();
        $exists = DB::table('wishlists')
            ->where('user_id', $user->id)
            ->where('product_id', $productId)
            ->exists();

        return response()->json(['in_wishlist' => $exists]);
    }
}
