<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Coupon extends Model
{
    use HasFactory;

    protected $fillable = [
        'code', 'type', 'value', 'min_order_amount', 'max_discount',
        'usage_limit', 'used_count', 'per_user_limit',
        'starts_at', 'expires_at', 'is_active', 'description',
    ];

    protected $casts = [
        'value' => 'decimal:2',
        'min_order_amount' => 'decimal:2',
        'max_discount' => 'decimal:2',
        'is_active' => 'boolean',
        'starts_at' => 'datetime',
        'expires_at' => 'datetime',
    ];

    public function scopeActive($query)
    {
        return $query->where('is_active', true)
            ->where(function ($q) {
            $q->whereNull('expires_at')
                ->orWhere('expires_at', '>', now());
        })
            ->where(function ($q) {
            $q->whereNull('starts_at')
                ->orWhere('starts_at', '<=', now());
        });
    }

    public function isValid(): bool
    {
        if (!$this->is_active)
            return false;
        if ($this->starts_at && $this->starts_at->isFuture())
            return false;
        if ($this->expires_at && $this->expires_at->isPast())
            return false;
        if ($this->usage_limit && $this->used_count >= $this->usage_limit)
            return false;
        return true;
    }

    public function calculateDiscount(float $orderTotal): float
    {
        if (!$this->isValid())
            return 0;
        if ($this->min_order_amount && $orderTotal < $this->min_order_amount)
            return 0;

        $discount = match ($this->type) {
                'percentage' => $orderTotal * ($this->value / 100),
                'fixed' => $this->value,
                'free_shipping' => 0,
                default => 0,
            };

        if ($this->max_discount && $discount > $this->max_discount) {
            $discount = $this->max_discount;
        }

        return round(min($discount, $orderTotal), 2);
    }

    public function incrementUsage(): void
    {
        $this->increment('used_count');
    }
}
