<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    protected $fillable = [
        'user_id',
        'number',
        'total_price',
        'status',
        'payment_status',
        'shipping_address',
        'notes',
        'customer_first_name',
        'customer_last_name',
        'customer_email',
        'customer_phone',
        'address',
        'city',
        'postal_code',
        'payment_method',
    ];

    protected $casts = [
        'total_price' => 'decimal:2',
        'created_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($order) {
            if (empty($order->number)) {
                $order->number = 'ORD-' . strtoupper(uniqid());
            }
        });

        static::updated(function ($order) {
            if ($order->isDirty('status') && $order->status === 'delivered') {
                $recipient = User::where('email', 'admin@example.com')->first();
                if ($recipient) {
                    \Filament\Notifications\Notification::make()
                        ->title('Order Delivered')
                        ->body("Order #{$order->number} has been marked as delivered.")
                        ->success()
                        ->sendToDatabase($recipient);
                }
            }
        });
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function scopeToday($query)
    {
        return $query->whereDate('created_at', today());
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function markAsCompleted()
    {
        $this->update(['status' => 'completed']);
    }
}
