<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    use HasFactory;

    protected $fillable = [
        'category_id',
        'brand_id',
        'name',
        'slug',
        'description',
        'short_description',
        'price',
        'sale_price',
        'sale_price_start',
        'sale_price_end',
        'stock',
        'low_stock_warning',
        'sku',
        'image',
        'is_featured',
        'is_active',
        'on_sale',
    ];

    protected $casts = [
        'is_featured' => 'boolean',
        'is_active' => 'boolean',
        'on_sale' => 'boolean',
        'price' => 'decimal:2',
        'sale_price' => 'decimal:2',
        'sale_price_start' => 'datetime',
        'sale_price_end' => 'datetime',
        'low_stock_warning' => 'integer',
        'stock' => 'integer',
    ];

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    public function variations()
    {
        return $this->hasMany(ProductVariation::class);
    }

    public function attributes()
    {
        return $this->belongsToMany(Attribute::class , 'product_attribute')
            ->withPivot('value')
            ->withTimestamps();
    }

    public function orderItems()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function isLowStock(): bool
    {
        return $this->stock <= ($this->low_stock_warning ?? 10);
    }

    public function getActivePriceAttribute()
    {
        if ($this->sale_price && $this->on_sale) {
            $now = now();
            if ($this->sale_price_start && $this->sale_price_end) {
                if ($now->between($this->sale_price_start, $this->sale_price_end)) {
                    return $this->sale_price;
                }
            }
            elseif ($this->sale_price_start) {
                if ($now->greaterThanOrEqualTo($this->sale_price_start)) {
                    return $this->sale_price;
                }
            }
            elseif ($this->sale_price_end) {
                if ($now->lessThanOrEqualTo($this->sale_price_end)) {
                    return $this->sale_price;
                }
            }
            else {
                // No dates set but sale price exists
                return $this->sale_price;
            }
        }
        return $this->price;
    }

    public function decreaseStock(int $quantity): void
    {
        $this->decrement('stock', $quantity);
    }

    public function increaseStock(int $quantity): void
    {
        $this->increment('stock', $quantity);
    }
}
