<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration 
{
    public function up(): void
    {
        // Coupons
        if (!Schema::hasTable('coupons')) {
            Schema::create('coupons', function (Blueprint $table) {
                $table->id();
                $table->string('code')->unique();
                $table->enum('type', ['percentage', 'fixed', 'free_shipping'])->default('percentage');
                $table->decimal('value', 10, 2)->default(0);
                $table->decimal('min_order_amount', 10, 2)->nullable();
                $table->decimal('max_discount', 10, 2)->nullable();
                $table->integer('usage_limit')->nullable();
                $table->integer('used_count')->default(0);
                $table->integer('per_user_limit')->nullable();
                $table->dateTime('starts_at')->nullable();
                $table->dateTime('expires_at')->nullable();
                $table->boolean('is_active')->default(true);
                $table->text('description')->nullable();
                $table->timestamps();
            });
        }

        // Flash Sales
        if (!Schema::hasTable('flash_sales')) {
            Schema::create('flash_sales', function (Blueprint $table) {
                $table->id();
                $table->string('title');
                $table->text('description')->nullable();
                $table->dateTime('starts_at');
                $table->dateTime('ends_at');
                $table->boolean('is_active')->default(true);
                $table->timestamps();
            });
        }

        // Flash Sale Products pivot
        if (!Schema::hasTable('flash_sale_products')) {
            Schema::create('flash_sale_products', function (Blueprint $table) {
                $table->id();
                $table->foreignId('flash_sale_id')->constrained()->cascadeOnDelete();
                $table->foreignId('product_id')->constrained()->cascadeOnDelete();
                $table->decimal('sale_price', 10, 2);
                $table->integer('stock_limit')->nullable();
                $table->integer('sold_count')->default(0);
                $table->timestamps();
            });
        }

        // Newsletter Subscribers
        if (!Schema::hasTable('newsletter_subscribers')) {
            Schema::create('newsletter_subscribers', function (Blueprint $table) {
                $table->id();
                $table->string('email')->unique();
                $table->string('name')->nullable();
                $table->boolean('is_active')->default(true);
                $table->timestamp('subscribed_at')->useCurrent();
                $table->timestamp('unsubscribed_at')->nullable();
                $table->timestamps();
            });
        }

        // Shipping Methods
        if (!Schema::hasTable('shipping_methods')) {
            Schema::create('shipping_methods', function (Blueprint $table) {
                $table->id();
                $table->string('name');
                $table->string('type')->default('flat_rate'); // flat_rate, free, weight_based, price_based
                $table->decimal('cost', 10, 2)->default(0);
                $table->decimal('min_order_amount', 10, 2)->nullable();
                $table->text('description')->nullable();
                $table->boolean('is_active')->default(true);
                $table->integer('sort_order')->default(0);
                $table->timestamps();
            });
        }

        // Tax Rates
        if (!Schema::hasTable('tax_rates')) {
            Schema::create('tax_rates', function (Blueprint $table) {
                $table->id();
                $table->string('name');
                $table->decimal('rate', 5, 2); // percentage
                $table->string('country')->nullable();
                $table->string('state')->nullable();
                $table->string('type')->default('percentage'); // percentage, fixed
                $table->boolean('is_active')->default(true);
                $table->integer('priority')->default(0);
                $table->timestamps();
            });
        }

        // Settings
        if (!Schema::hasTable('settings')) {
            Schema::create('settings', function (Blueprint $table) {
                $table->id();
                $table->string('group')->default('general');
                $table->string('key')->unique();
                $table->text('value')->nullable();
                $table->timestamps();
            });
        }

        // Enhance orders table
        if (Schema::hasTable('orders')) {
            Schema::table('orders', function (Blueprint $table) {
                if (!Schema::hasColumn('orders', 'payment_method')) {
                    $table->string('payment_method')->nullable()->after('payment_status');
                }
                if (!Schema::hasColumn('orders', 'tracking_number')) {
                    $table->string('tracking_number')->nullable()->after('payment_method');
                }
                if (!Schema::hasColumn('orders', 'billing_address')) {
                    $table->text('billing_address')->nullable()->after('shipping_address');
                }
                if (!Schema::hasColumn('orders', 'subtotal')) {
                    $table->decimal('subtotal', 15, 2)->nullable()->after('total_price');
                }
                if (!Schema::hasColumn('orders', 'shipping_cost')) {
                    $table->decimal('shipping_cost', 10, 2)->default(0)->after('subtotal');
                }
                if (!Schema::hasColumn('orders', 'tax_amount')) {
                    $table->decimal('tax_amount', 10, 2)->default(0)->after('shipping_cost');
                }
                if (!Schema::hasColumn('orders', 'discount_amount')) {
                    $table->decimal('discount_amount', 10, 2)->default(0)->after('tax_amount');
                }
                if (!Schema::hasColumn('orders', 'coupon_code')) {
                    $table->string('coupon_code')->nullable()->after('discount_amount');
                }
            });
        }

        // Enhance vendors table
        if (Schema::hasTable('vendors')) {
            Schema::table('vendors', function (Blueprint $table) {
                if (!Schema::hasColumn('vendors', 'commission_rate')) {
                    $table->decimal('commission_rate', 5, 2)->default(10)->after('is_active');
                }
                if (!Schema::hasColumn('vendors', 'balance')) {
                    $table->decimal('balance', 15, 2)->default(0)->after('commission_rate');
                }
                if (!Schema::hasColumn('vendors', 'total_earned')) {
                    $table->decimal('total_earned', 15, 2)->default(0)->after('balance');
                }
                if (!Schema::hasColumn('vendors', 'verification_status')) {
                    $table->string('verification_status')->default('pending')->after('total_earned');
                }
                if (!Schema::hasColumn('vendors', 'banner')) {
                    $table->string('banner')->nullable()->after('logo');
                }
            });
        }
    }

    public function down(): void
    {
        Schema::dropIfExists('settings');
        Schema::dropIfExists('tax_rates');
        Schema::dropIfExists('shipping_methods');
        Schema::dropIfExists('newsletter_subscribers');
        Schema::dropIfExists('flash_sale_products');
        Schema::dropIfExists('flash_sales');
        Schema::dropIfExists('coupons');

        if (Schema::hasTable('orders')) {
            Schema::table('orders', function (Blueprint $table) {
                $columns = ['payment_method', 'tracking_number', 'billing_address', 'subtotal', 'shipping_cost', 'tax_amount', 'discount_amount', 'coupon_code'];
                foreach ($columns as $col) {
                    if (Schema::hasColumn('orders', $col)) {
                        $table->dropColumn($col);
                    }
                }
            });
        }

        if (Schema::hasTable('vendors')) {
            Schema::table('vendors', function (Blueprint $table) {
                $columns = ['commission_rate', 'balance', 'total_earned', 'verification_status', 'banner'];
                foreach ($columns as $col) {
                    if (Schema::hasColumn('vendors', $col)) {
                        $table->dropColumn($col);
                    }
                }
            });
        }
    }
};
