<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class DashboardSeeder extends Seeder
{
    public function run()
    {
        $this->command->info('Step 1: Admin User');
        $adminId = DB::table('users')->where('email', 'admin@example.com')->value('id');
        if (!$adminId) {
            $adminId = DB::table('users')->insertGetId([
                'name' => 'Admin User',
                'email' => 'admin@example.com',
                'password' => Hash::make('password'),
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
        $this->command->info("✅ Admin ID: $adminId");

        $this->command->info('Step 2: Attributes & Categories');
        $sizeId = DB::table('attributes')->where('name', 'Size')->value('id');
        if (!$sizeId) {
            $sizeId = DB::table('attributes')->insertGetId(['name' => 'Size', 'type' => 'select', 'created_at' => now(), 'updated_at' => now()]);
        }

        $colorId = DB::table('attributes')->where('name', 'Color')->value('id');
        if (!$colorId) {
            $colorId = DB::table('attributes')->insertGetId(['name' => 'Color', 'type' => 'select', 'created_at' => now(), 'updated_at' => now()]);
        }

        // Create Category (Required for Products) - REMOVED description
        $catId = DB::table('categories')->where('slug', 'luxury-collection')->value('id');
        if (!$catId) {
            $catId = DB::table('categories')->insertGetId([
                'name' => 'Luxury Collection',
                'slug' => 'luxury-collection',
                'is_active' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
        $this->command->info("✅ Category Created");

        $this->command->info('Step 3: Products & Variations');
        for ($i = 1; $i <= 5; $i++) {
            $prodSlug = 'luxury-product-' . $i . '-' . uniqid();
            $prodId = DB::table('products')->insertGetId([
                'name' => 'Luxury Product ' . $i,
                'slug' => $prodSlug,
                'description' => 'A premium product for testing.',
                'price' => rand(100, 500),
                'stock' => 50,
                'category_id' => $catId,
                'is_active' => true,
                'is_featured' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Variation
            DB::table('product_variations')->insert([
                'product_id' => $prodId,
                'attribute_id' => $sizeId,
                'value' => '42',
                'sku' => 'SKU-' . $prodId . '-42',
                'price_adjustment' => 0,
                'stock' => 20,
                'is_active' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Create 1 Order per Product
            $orderId = DB::table('orders')->insertGetId([
                'user_id' => $adminId,
                'number' => 'ORD-' . rand(100000, 999999),
                'status' => 'completed',
                'total_price' => 100,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            DB::table('order_items')->insert([
                'order_id' => $orderId,
                'product_id' => $prodId,
                'quantity' => 1,
                'unit_price' => 100,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Create Banner if not exists
            if (DB::table('banners')->count() == 0) {
                DB::table('banners')->insert([
                    [
                        'title' => 'Experience Aether',
                        'description' => 'Elevate your lifestyle with our premium collection.',
                        'image' => 'banners/hero-placeholder.jpg',
                        'button_text' => 'Shop Now',
                        'button_url' => '/products',
                        'is_active' => true,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]
                ]);
            }
        }

        // Create Eid Collection Category
        $eidCatId = DB::table('categories')->where('slug', 'eid-2026')->value('id');
        if (!$eidCatId) {
            $eidCatId = DB::table('categories')->insertGetId([
                'name' => 'EID/26 Collection',
                'slug' => 'eid-2026',
                'description' => 'Celebrate Eid in style with our exclusive collection.',
                'is_active' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }

        // Add 6 Eid Products
        $eidProducts = [
            ['name' => 'Midnight Blue Panjabi', 'price' => 3500],
            ['name' => 'Golden Embroidered Sheriff', 'price' => 5500],
            ['name' => 'Silk Saree - Ruby Red', 'price' => 8500],
            ['name' => 'Emerald Green Palazzo Set', 'price' => 4200],
            ['name' => 'Premium Kabli Set', 'price' => 4000],
            ['name' => 'Kids Festive Kurta', 'price' => 1200],
        ];

        foreach ($eidProducts as $idx => $prod) {
            DB::table('products')->insertGetId([
                'name' => $prod['name'],
                'slug' => Str::slug($prod['name']) . '-' . uniqid(),
                'description' => 'Exclusive Eid collection item. Limited stock available.',
                'price' => $prod['price'],
                'stock' => 20,
                'category_id' => $eidCatId,
                'is_active' => true,
                'is_featured' => $idx < 3, // Feature first 3
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }

        $this->command->info("✅ Eid Collection & Products Created");
        $this->command->info("✅ Products, Variations & Orders Created");
    }
}
