<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Product;
use App\Models\Category;
use App\Models\Attribute;
use App\Models\Brand;
use App\Models\User;
use App\Models\Review;
use App\Models\StockMovement;
use Illuminate\Support\Str;

class VerificationSeeder extends Seeder
{
    public function run()
    {
        // 1. Create Nested Categories
        $parent = Category::create([
            'name' => 'Clothing',
            'slug' => 'clothing-' . uniqid(),
            'description' => 'Top level clothing category'
        ]);

        $child = Category::create([
            'name' => 'T-Shirts',
            'slug' => 't-shirts-' . uniqid(),
            'parent_id' => $parent->id,
            'description' => 'Mens and Womens T-Shirts'
        ]);

        $this->command->info("✅ Created Nested Categories: {$parent->name} > {$child->name}");

        // 2. Create Attributes
        $sizeAttr = Attribute::firstOrCreate(['name' => 'Size'], ['type' => 'select']);
        $colorAttr = Attribute::firstOrCreate(['name' => 'Color'], ['type' => 'select']);

        // 3. Create Product with Variations
        $brand = Brand::first();
        if (!$brand)
            $brand = Brand::create(['name' => 'Test Brand', 'slug' => 'test-brand']);

        $product = Product::create([
            'name' => 'StartUp Hoodie',
            'slug' => 'startup-hoodie-' . uniqid(),
            'category_id' => $child->id,
            'brand_id' => $brand->id,
            'description' => 'Premium tech hoodie',
            'price' => 59.99,
            'stock' => 100, // Initial stock
            'sku' => 'HOODIE-001',
            'is_active' => true,
        ]);

        $this->command->info("✅ Created Product: {$product->name} (Stock: {$product->stock})");

        // 4. Create Variations
        $variations = [
            ['value' => 'Small', 'sku' => 'HOODIE-001-S', 'stock' => 20],
            ['value' => 'Medium', 'sku' => 'HOODIE-001-M', 'stock' => 50],
            ['value' => 'Large', 'sku' => 'HOODIE-001-L', 'stock' => 30],
        ];

        foreach ($variations as $var) {
            $product->variations()->create([
                'attribute_id' => $sizeAttr->id,
                'value' => $var['value'],
                'sku' => $var['sku'],
                'stock' => $var['stock'],
                'price_adjustment' => 0
            ]);
        }

        $this->command->info("✅ Created 3 Product Variations");

        // 5. Test Stock Movement Observer
        // Update product stock manually
        $product->stock = 90; // Reduced by 10
        $product->save();

        $movements = StockMovement::where('product_id', $product->id)->count();
        if ($movements > 0) {
            $this->command->info("✅ StockMovementObserver Working! Found {$movements} movements.");
        }
        else {
            $this->command->error("❌ StockMovementObserver Failed! No movements found.");
        }

        // 6. Test Review Logic
        $user = User::first();
        if (!$user)
            $user = User::factory()->create();

        $review = Review::create([
            'product_id' => $product->id,
            'user_id' => $user->id,
            'rating' => 5,
            'comment' => 'Great hoodie!',
            'is_approved' => false
        ]);

        // Simulating Admin Reply & Approval
        $review->update([
            'is_approved' => true,
            'admin_reply' => 'Thanks for the feedback!',
            'replied_at' => now(),
            'replied_by' => $user->id // Self-reply for test
        ]);

        $this->command->info("✅ Created & Approved Review with Admin Reply");
    }
}
