import { defineStore } from 'pinia';
import { ref, computed } from 'vue';
import axios from 'axios';

export const useAuthStore = defineStore('auth', () => {
    // State
    const user = ref(null);
    const token = ref(localStorage.getItem('token') || null);
    const isLoading = ref(false);

    // Getters
    const isAuthenticated = computed(() => !!token.value);
    const userName = computed(() => user.value?.name || 'Guest');

    // Actions
    async function login(credentials) {
        isLoading.value = true;
        try {
            const response = await axios.post('/api/login', credentials);
            token.value = response.data.token;
            user.value = response.data.user;
            localStorage.setItem('token', token.value);
            axios.defaults.headers.common['Authorization'] = `Bearer ${token.value}`;
            return true;
        } catch (error) {
            console.error('Login failed:', error);
            return false;
        } finally {
            isLoading.value = false;
        }
    }

    async function register(userData) {
        isLoading.value = true;
        try {
            const response = await axios.post('/api/register', userData);
            token.value = response.data.token;
            user.value = response.data.user;
            localStorage.setItem('token', token.value);
            axios.defaults.headers.common['Authorization'] = `Bearer ${token.value}`;
            return true;
        } catch (error) {
            console.error('Registration failed:', error);
            return false;
        } finally {
            isLoading.value = false;
        }
    }

    function logout() {
        user.value = null;
        token.value = null;
        localStorage.removeItem('token');
        delete axios.defaults.headers.common['Authorization'];
    }

    async function fetchUser() {
        if (!token.value) return;

        isLoading.value = true;
        try {
            const response = await axios.get('/api/user');
            user.value = response.data;
        } catch (error) {
            console.error('Failed to fetch user:', error);
            logout();
        } finally {
            isLoading.value = false;
        }
    }

    async function updateProfile(userData) {
        isLoading.value = true;
        try {
            const response = await axios.post('/api/user/update', userData);
            user.value = response.data.user;
            return { success: true, message: response.data.message };
        } catch (error) {
            console.error('Update profile failed:', error);
            const message = error.response?.data?.message || 'Failed to update profile';
            return { success: false, message };
        } finally {
            isLoading.value = false;
        }
    }

    // Initialize
    if (token.value) {
        axios.defaults.headers.common['Authorization'] = `Bearer ${token.value}`;
        fetchUser();
    }

    return {
        user,
        token,
        isLoading,
        isAuthenticated,
        userName,
        login,
        register,
        logout,
        fetchUser,
        updateProfile,
    };
});
