import { defineStore } from 'pinia';
import { ref, computed } from 'vue';

export const useCartStore = defineStore('cart', () => {
    // State
    const items = ref([]);
    const isLoading = ref(false);

    // Getters
    const itemCount = computed(() => {
        return items.value.reduce((total, item) => total + item.quantity, 0);
    });

    const subtotal = computed(() => {
        return items.value.reduce((total, item) => {
            return total + (item.price * item.quantity);
        }, 0);
    });

    const total = computed(() => {
        // Add tax, shipping, discounts here
        return subtotal.value;
    });

    // Actions
    function addItem(product, quantity = 1) {
        const existingItem = items.value.find(item => item.id === product.id);

        if (existingItem) {
            existingItem.quantity += quantity;
        } else {
            items.value.push({
                id: product.id,
                name: product.name,
                price: product.price,
                image: product.image,
                quantity: quantity,
            });
        }

        saveToLocalStorage();
    }

    function removeItem(productId) {
        const index = items.value.findIndex(item => item.id === productId);
        if (index > -1) {
            items.value.splice(index, 1);
            saveToLocalStorage();
        }
    }

    function updateQuantity(productId, quantity) {
        const item = items.value.find(item => item.id === productId);
        if (item) {
            item.quantity = quantity;
            if (item.quantity <= 0) {
                removeItem(productId);
            } else {
                saveToLocalStorage();
            }
        }
    }

    function clearCart() {
        items.value = [];
        saveToLocalStorage();
    }

    function saveToLocalStorage() {
        localStorage.setItem('cart', JSON.stringify(items.value));
    }

    function loadFromLocalStorage() {
        const savedCart = localStorage.getItem('cart');
        if (savedCart) {
            items.value = JSON.parse(savedCart);
        }
    }

    // Initialize cart from localStorage
    loadFromLocalStorage();

    async function checkout(orderData) {
        isLoading.value = true;
        try {
            const response = await fetch('/api/checkout', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                },
                body: JSON.stringify({
                    items: items.value,
                    ...orderData
                }),
            });

            if (!response.ok) {
                const errorData = await response.json();
                const error = new Error(errorData.message || 'Checkout failed');
                error.details = errorData.error || null;
                throw error;
            }

            const data = await response.json();
            clearCart();
            return data;
        } catch (error) {
            console.error('Checkout error:', error);
            throw error;
        } finally {
            isLoading.value = false;
        }
    }

    return {
        items,
        isLoading,
        itemCount,
        subtotal,
        total,
        addItem,
        removeItem,
        updateQuantity,
        clearCart,
        checkout,
    };
});
