import { defineStore } from 'pinia';
import { useAuthStore } from './auth';
import { useToast } from 'vue-toastification';

export const useWishlistStore = defineStore('wishlist', {
    state: () => ({
        items: [],
        loading: false,
    }),

    getters: {
        count: (state) => state.items.length,
        isInWishlist: (state) => (productId) => {
            return state.items.some(item => item.id === productId);
        }
    },

    actions: {
        async fetchWishlist() {
            const authStore = useAuthStore();
            if (!authStore.isAuthenticated) {
                this.items = [];
                return;
            }

            this.loading = true;
            try {
                const response = await fetch('/api/wishlist', {
                    headers: {
                        'Authorization': `Bearer ${localStorage.getItem('token')}`,
                        'Accept': 'application/json'
                    }
                });

                if (response.ok) {
                    this.items = await response.json();
                }
            } catch (error) {
                console.error('Failed to fetch wishlist', error);
            } finally {
                this.loading = false;
            }
        },

        async addToWishlist(product) {
            const authStore = useAuthStore();
            const toast = useToast();

            if (!authStore.isAuthenticated) {
                toast.warning('Please login to add items to wishlist');
                return false;
            }

            // Optimistic update
            if (this.isInWishlist(product.id)) {
                toast.info('Product already in wishlist');
                return true;
            }

            try {
                const response = await fetch('/api/wishlist', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${localStorage.getItem('token')}`,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ product_id: product.id })
                });

                if (response.ok) {
                    this.items.push(product); // Add to local state
                    toast.success('Added to wishlist');
                    return true;
                }
            } catch (error) {
                console.error('Failed to add to wishlist', error);
                toast.error('Failed to add to wishlist');
            }
            return false;
        },

        async removeFromWishlist(productId) {
            const authStore = useAuthStore();
            const toast = useToast();

            if (!authStore.isAuthenticated) return;

            try {
                const response = await fetch(`/api/wishlist/${productId}`, {
                    method: 'DELETE',
                    headers: {
                        'Authorization': `Bearer ${localStorage.getItem('token')}`,
                        'Accept': 'application/json'
                    }
                });

                if (response.ok) {
                    this.items = this.items.filter(item => item.id !== productId);
                    toast.success('Removed from wishlist');
                }
            } catch (error) {
                console.error('Failed to remove from wishlist', error);
                toast.error('Failed to remove from wishlist');
            }
        },

        async toggleWishlist(product) {
            if (this.isInWishlist(product.id)) {
                await this.removeFromWishlist(product.id);
            } else {
                await this.addToWishlist(product);
            }
        }
    }
});
